/*

An HTTP Request example                     Notes

GET /path/file.html  HTTP/1.0               whitespace required
Accept:  text/html                          request header fields
Accept:  audio/x                            tell the client which data types
User-agent:  MacWeb                         can be handled by the browser


HTTP General form

<method> <resource identifier> <HTTP Version> <crlf>
[<Header> : <value>] <crlf>
 
:  :   :
[<Header> : <value>] <crlf>
    a blank line
[entity body]


The resource identifier field specifies the name of the target
resource; it's the URL stripped of the protocol and the server
domain name. When using the GET method, this field will also
contain a series of name=value pairs separated by &. When using
a POST method, the entity body contains these pairs.

The HTTP version identifies the protocol used by the client.

*/
 
// Adapted from Core Servlets and JavaServerPages
// by Marty Hall, chapter 16

import java.net.*;
import java.io.*;
import java.util.StringTokenizer;

public class EchoServer extends NetworkServer {

   protected int maxRequestLines = 50;     // Post data is brought in 
                                           // as a single string.
   protected String serverName = "EchoServer";

   public static void main(String a[]) {

       int port = 6502;
       new EchoServer(port,0);     // loop forever
   }

   public EchoServer(int port, int maxConnections) {
       super(port,maxConnections); // call base class constructor 
       listen();                   // call base class listen()                 
   }                               // listen calls handleConnection()
   
   // Overrides base class handleConection and is called by listen()
   public void handleConnection(Socket server) throws IOException {


          // Assign readers and writers to the socket      
          BufferedReader in = new BufferedReader(
                                  new InputStreamReader(
                                      server.getInputStream() ));

          PrintWriter out = new PrintWriter(server.getOutputStream(),true);

          // Announce connection to console
          System.out.println(serverName + " got connection from "+
                                 server.getInetAddress().getHostName() + "\n");
                                
          
          String inputLines[] = new String[maxRequestLines];
          int i;

          for(i = 0; i < maxRequestLines; i++) {

                   inputLines[i] = in.readLine();
                   if(inputLines[i] == null) break; // client closed connection

                   if(inputLines[i].length() == 0) {  // blank line
                                                      // maybe done or maybe post
                             if(usingPost(inputLines)) {
                                  // readPostData reads into a single string
                                  // at location i+1
                                  readPostData(inputLines,i,in);
                                  // i was not changed in the procedure so
                                  // bump it one past the post data string
                                  i = i + 2;
                              }

                              break;
                   }
           }
          
           printHeader(out);
           for(int j = 0; j < i; j++) {
                out.println(inputLines[j]);
           }

           
           printTrailer(out);
           server.close();
    }


    private void printHeader(PrintWriter out) {


          out.println(
                 "HTTP/1.0 200 OK\r\n"            +
                 "Server: " + serverName + "\r\n" +
                 "Content-Type: text/html\r\n"    +
                 "\r\n"                           +
                 "<!DOCTYPE HTML PUBLIC "         +
                 "\"-//W3C//DTD HTML 4.0 Transitional//EN\">\n" +
                 "<HTML>\n"                       +
                 "<HEAD>\n"                       +
                 "   <TITLE>" + serverName + " Results</TITLE>\n" +
                 "</HEAD>\n"                      +
                 "\n"                             +
                 "<BODY BGCOLOR=\"#FDF5E6\">\n"   +
                 "<H1 ALIGN=\"CENTER\">" + serverName + " Results</H1>\n" +
                 "Here is your request line and request headers\n" +
                 "sent by your browser:\n" + 
                 "<PRE>"
           );      
    }

    private void printTrailer(PrintWriter out) {

           out.println("</PRE>\n"   +
                       "</BODY>\n"  +
                       "</HTML>\n");
    }

    private boolean usingPost(String[] inputs) {

           return (inputs[0].toUpperCase().startsWith("POST"));

    }

    // Read the post data a a single array of char and place it all
    // in one string.
    
    private void readPostData (String inputs[], int i, BufferedReader in) 
                              throws IOException {

           int contentLength = contentLength(inputs);
           char postData[] = new char[contentLength];
           in.read(postData, 0, contentLength);
           
           // All of the post data is converted to a single string
           inputs[++i] = new String(postData,0,contentLength);
    }

    // The header fields may arrive in any order.
    // Search for and return the CONTENT-LENGTH.
    private int contentLength(String inputs[]) {

           String input;
           for(int i = 0; i < inputs.length; i++) {
              if(inputs[i].length() == 0) break;
              input = inputs[i].toUpperCase();
              if(input.startsWith("CONTENT-LENGTH")) return (getLength(input));
           }
           return (0);
    }

    // Return the integer associated with the second token.
    private int getLength(String length) {

          StringTokenizer tok = new StringTokenizer(length);
          tok.nextToken();
          return (Integer.parseInt(tok.nextToken()));
    }
}



           

                      